<?php
    /**
     * Created by PhpStorm.
     * User: Admin
     * Date: 7/30/2019
     * Time: 1:56 PM
     */

    namespace Modules\Flight\Admin;

    use Illuminate\Http\Request;
    use Illuminate\Support\Facades\Auth;
    use Illuminate\Support\Facades\Validator;
    use Modules\AdminController;
    use Modules\Core\Events\CreatedServicesEvent;
    use Modules\Core\Events\UpdatedServiceEvent;
    use Modules\Core\Models\Attributes;
    use Modules\Location\Models\Location;
    use Modules\Location\Models\LocationCategory;
    use Modules\Flight\Models\Flight;
    use Modules\Flight\Models\FlightTerm;

    class FlightController extends AdminController
    {
        protected $space;
        protected $flight_term;
        protected $attributes;
        protected $location;
        /**
         * @var string
         */
        private $locationCategoryClass;
        /**
         * @var string
         */
        private $flight;

        public function __construct()
        {
            $this->setActiveMenu(route('flight.admin.index'));
            $this->flight = Flight::class;
            $this->flight_term = FlightTerm::class;
            $this->attributes = Attributes::class;
            $this->location = Location::class;
            $this->locationCategoryClass = LocationCategory::class;
        }

        public function callAction($method, $parameters)
        {
            if (!Flight::isEnable()) {
                return redirect('/');
            }
            return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
        }

        public function index(Request $request)
        {
            $this->checkPermission('flight_view');
            $query = $this->flight::query();
            $query->orderBy('id', 'desc');
            if (!empty($flight_name = $request->input('s'))) {
                $query->where('title', 'LIKE', '%'.$flight_name.'%')->orWhere('code','like', '%'.$flight_name.'%');
                $query->orderBy('title', 'asc');
            }

            if ($this->hasPermission('flight_manage_others')) {
                if (!empty($author = $request->input('vendor_id'))) {
                    $query->where('author_id', $author);
                }
            } else {
                $query->where('author_id', Auth::id());
            }
            $data = [
                'row'                  => new $this->flight,
                'rows'                 => $query->with(['airline','airportTo','airportFrom','author'])->paginate(20),
                'flight_manage_others' => $this->hasPermission('flight_manage_others'),
                'breadcrumbs'          => [
                    [
                        'name' => __('Flights'),
                        'url'  => route('flight.admin.index')
                    ],
                    [
                        'name'  => __('All'),
                        'class' => 'active'
                    ],
                ],
                'page_title'           => __("Flight Management")
            ];
            return view('Flight::admin.index', $data);
        }

        public function recovery(Request $request)
        {
            $this->checkPermission('flight_view');
            $query = $this->flight::onlyTrashed();
            $query->orderBy('id', 'desc');
            if (!empty($flight_name = $request->input('s'))) {
                $query->where('title', 'LIKE', '%'.$flight_name.'%');
                $query->orderBy('title', 'asc');
            }

            if ($this->hasPermission('flight_manage_others')) {
                if (!empty($author = $request->input('vendor_id'))) {
                    $query->where('author_id', $author);
                }
            } else {
                $query->where('author_id', Auth::id());
            }
            $data = [
                'rows'                 => $query->with(['author'])->paginate(20),
                'flight_manage_others' => $this->hasPermission('flight_manage_others'),
                'recovery'             => 1,
                'breadcrumbs'          => [
                    [
                        'name' => __('Flights'),
                        'url'  => route('space.admin.index')
                    ],
                    [
                        'name'  => __('Recovery'),
                        'class' => 'active'
                    ],
                ],
                'page_title'           => __("Recovery Flight Management")
            ];
            return view('Flight::admin.flight.index', $data);
        }

    public function create(Request $request)
    {
        $this->checkPermission('flight_create');
        $row = new $this->flight();
        $row->fill([
            'status' => 'publish'
        ]);
        $data = [
            'row'            => $row,
            'attributes'     => $this->attributes::where('service', 'flight')->get(),
            'translation'    => $row,
            'breadcrumbs'    => [
                [
                    'name' => __('Flights'),
                    'url'  => route('flight.admin.index')
                ],
                [
                    'name'  => __('Add Flight'),
                    'class' => 'active'
                ],
            ],
            'page_title'     => __("Add new Flight")
        ];
        return view('Flight::admin.detail', $data);
    }

        public function edit(Request $request, $id)
        {
            $this->checkPermission('flight_update');
            $row = $this->flight::with(['airline','airportTo','airportFrom'])->find($id);
            if (empty($row)) {
                return redirect(route('flight.admin.index'));
            }
            if (!$this->hasPermission('flight_manage_others')) {
                if ($row->author_id != Auth::id()) {
                    return redirect(route('flight.admin.index'));
                }
            }
            $data = [
                'row'               => $row,
                "selected_terms"    => $row->terms->pluck('term_id'),
                'attributes'        => $this->attributes::where('service', 'flight')->get(),
                'enable_multi_lang' => true,
                'breadcrumbs'       => [
                    [
                        'name' => __('Flights'),
                        'url'  => route('flight.admin.index')
                    ],
                    [
                        'name'  => __('Edit Flight'),
                        'class' => 'active'
                    ],
                ],
                'page_title'        => __("Edit: #:name", ['name' => $row->id])
            ];
            return view('Flight::admin.detail', $data);
        }

        public function store(Request $request, $id)
        {

            if ($id > 0) {
                $this->checkPermission('flight_update');
                $row = $this->flight::find($id);
                if (empty($row)) {
                    return redirect(route('flight.admin.index'));
                }

                if ($row->author_id != Auth::id() and !$this->hasPermission('flight_manage_others')) {
                    return redirect(route('flight.admin.index'));
                }
            } else {
                $this->checkPermission('flight_create');
                $row = new $this->flight();
                $row->status = "publish";
            }


            $validator = Validator::make($request->all(), [
                'departure_time'=>'required',
                'arrival_time'=>'required',
                'duration'=>'required',
                'airport_from'=>'required',
                'airport_to'=>'required',
                'airline_id'=>'required',
            ]);
            if ($validator->fails()) {
                return redirect()->back()->with(['errors' => $validator->errors()]);
            }
            $dataKeys = [
                'title',
                'code',
                'departure_time',
                'arrival_time',
                'duration',
                'airport_from',
                'airport_to',
                'airline_id',
                'status',
            ];
            if ($this->hasPermission('flight_manage_others')) {
                $dataKeys[] = 'author_id';
            }

            $row->fillByAttr($dataKeys, $request->input());
            $res = $row->save();
            if ($res) {
                $this->saveTerms($row,$request);
                if ($id > 0) {
                    return back()->with('success', __('Flight updated'));
                } else {
                    return redirect(route('flight.admin.edit', $row->id))->with('success', __('Flight created'));
                }
            }
        }
        public function saveTerms($row, $request)
        {
            $this->checkPermission('flight_manage_attributes');
            if (empty($request->input('terms'))) {
                $this->flight_term::where('target_id', $row->id)->delete();
            } else {
                $term_ids = $request->input('terms');
                foreach ($term_ids as $term_id) {
                    $this->flight_term::firstOrCreate([
                        'term_id'   => $term_id,
                        'target_id' => $row->id
                    ]);
                }
                $this->flight_term::where('target_id', $row->id)->whereNotIn('term_id', $term_ids)->delete();
            }
        }

        public function bulkEdit(Request $request)
        {

            $ids = $request->input('ids');
            $action = $request->input('action');
            if (empty($ids) or !is_array($ids)) {
                return redirect()->back()->with('error', __('No items selected!'));
            }
            if (empty($action)) {
                return redirect()->back()->with('error', __('Please select an action!'));
            }

            switch ($action) {
                case "delete":
                    foreach ($ids as $id) {
                        $query = $this->flight::where("id", $id);
                        if (!$this->hasPermission('flight_manage_others')) {
                            $query->where("create_user", Auth::id());
                            $this->checkPermission('flight_delete');
                        }
                        $row = $query->first();
                        if (!empty($row)) {
                            $row->delete();
                            event(new UpdatedServiceEvent($row));

                        }
                    }
                    return redirect()->back()->with('success', __('Deleted success!'));
                    break;
                case "permanently_delete":
                    foreach ($ids as $id) {
                        $query = $this->flight::where("id", $id);
                        if (!$this->hasPermission('flight_manage_others')) {
                            $query->where("create_user", Auth::id());
                            $this->checkPermission('flight_delete');
                        }
                        $row = $query->withTrashed()->first();
                        if ($row) {
                            $row->forceDelete();
                        }
                    }
                    return redirect()->back()->with('success', __('Permanently delete success!'));
                    break;
                case "recovery":
                    foreach ($ids as $id) {
                        $query = $this->flight::withTrashed()->where("id", $id);
                        if (!$this->hasPermission('flight_manage_others')) {
                            $query->where("create_user", Auth::id());
                            $this->checkPermission('flight_delete');
                        }
                        $row = $query->first();
                        if (!empty($row)) {
                            $row->restore();
                            event(new UpdatedServiceEvent($row));

                        }
                    }
                    return redirect()->back()->with('success', __('Recovery success!'));
                    break;
                case "clone":
                    $this->checkPermission('flight_create');
                    foreach ($ids as $id) {
                        (new $this->flight())->saveCloneByID($id);
                    }
                    return redirect()->back()->with('success', __('Clone success!'));
                    break;
                default:
                    // Change status
                    foreach ($ids as $id) {
                        $query = $this->flight::where("id", $id);
                        if (!$this->hasPermission('flight_manage_others')) {
                            $query->where("create_user", Auth::id());
                            $this->checkPermission('flight_update');
                        }
                        $row = $query->first();
                        $row->status = $action;
                        $row->save();
                        event(new UpdatedServiceEvent($row));
                    }
                    return redirect()->back()->with('success', __('Update success!'));
                    break;
            }


        }
    }
