<?php
namespace Modules\Vendor\Controllers;

use App\Helpers\ReCaptchaEngine;
use App\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\MessageBag;
use Illuminate\Validation\Rules\Password;
use Matrix\Exception;
use Modules\FrontendController;
use Modules\User\Events\NewVendorRegistered;
use Modules\User\Events\SendMailUserRegistered;
use Modules\Vendor\Models\VendorRequest;
use Modules\Booking\Models\Booking;
use Modules\ServiceProvider;
use Modules\Vendor\Models\VendorDomain;
use Modules\Vendor\Models\VendorModuleAccess;

class VendorController extends FrontendController
{
    protected $bookingClass;
    public function __construct()
    {
        $this->bookingClass = Booking::class;
        parent::__construct();
    }
    public function register(Request $request)
    {
        $rules = [
            'first_name' => [
                'required',
                'string',
                'max:255'
            ],
            'last_name'  => [
                'required',
                'string',
                'max:255'
            ],
            'business_name'  => [
                'required',
                'string',
                'max:255'
            ],
            'email'      => [
                'required',
                'string',
                'email',
                'max:255',
                'unique:users'
            ],
            'password'   => [
                'required',
                'string',
                Password::min(8)
                    ->mixedCase()
                    ->numbers()
                    ->symbols()
                    ->uncompromised(),
            ],
            'term'       => ['required'],
        ];
        $messages = [
            'email.required'      => __('Email is required field'),
            'email.email'         => __('Email invalidate'),
            'password.required'   => __('Password is required field'),
            'first_name.required' => __('The first name is required field'),
            'last_name.required'  => __('The last name is required field'),
            'business_name.required'  => __('The business name is required field'),
            'term.required'       => __('The terms and conditions field is required'),
        ];
        if (ReCaptchaEngine::isEnable() and setting_item("user_enable_register_recaptcha")) {
            $messages['g-recaptcha-response.required'] = __('Please verify the captcha');
            $rules['g-recaptcha-response'] = ['required'];
        }
        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return response()->json([
                'error'    => true,
                'messages' => $validator->errors()
            ], 200);
        } else {
            if (ReCaptchaEngine::isEnable() and setting_item("user_enable_register_recaptcha")) {
                $codeCapcha = $request->input('g-recaptcha-response');
                if (!ReCaptchaEngine::verify($codeCapcha)) {
                    $errors = new MessageBag(['message_error' => __('Please verify the captcha')]);
                    return response()->json([
                        'error'    => true,
                        'messages' => $errors
                    ], 200);
                }
            }
            $user = new \App\User();

            $user = $user->fill([
                'first_name'=>$request->input('first_name'),
                'last_name'=>$request->input('last_name'),
                'email'=>$request->input('email'),
                'password'=>Hash::make($request->input('password')),
                'business_name'=>$request->input('business_name'),
                'phone'=>$request->input('phone'),
            ]);
            $user->status = 'publish';

            $user->save();
            if (empty($user)) {
                return $this->sendError(__("Can not register"));
            }

            //                check vendor auto approved
            $vendorAutoApproved = setting_item('vendor_auto_approved');
            $dataVendor['role_request'] = setting_item('vendor_role');
            if ($vendorAutoApproved) {
                if ($dataVendor['role_request']) {
                    $user->assignRole($dataVendor['role_request']);
                }
                $dataVendor['status'] = 'approved';
                $dataVendor['approved_time'] = now();
            } else {
                $dataVendor['status'] = 'pending';
                $user->assignRole(setting_item('user_role'));
            }
            $vendorRequestData = $user->vendorRequest()->save(new VendorRequest($dataVendor));
            Auth::loginUsingId($user->id);
            try {
                event(new NewVendorRegistered($user, $vendorRequestData));
            } catch (Exception $exception) {
                Log::warning("NewVendorRegistered: " . $exception->getMessage());
            }
            if ($vendorAutoApproved) {
                return $this->sendSuccess([
                    'redirect' => url(app_get_locale(false, '/')),
                ]);
            } else {
                return $this->sendSuccess([
                    'redirect' => url(app_get_locale(false, '/')),
                ], __("Register success. Please wait for admin approval"));
            }
        }
    }
    public function domainRegister(){
        $user = auth()->user();
        $get_vendor_domain = VendorDomain::where('vendor_id',$user->id)->first();
        return view('Vendor::frontend.domainRegistration.index',compact('get_vendor_domain'));
    }

    public function domainStore(Request $request){
        $request->validate([
            'url_type' => 'required',
            'domain_name' => 'required|unique:vendor_domains',
        ]);

        // VendorDomain::storeDomain($request->all());

        $user = auth()->user();
        $data = new VendorDomain();
        $data->url = $request->url_type;
        $data->domain_name = $request->domain_name;
        $data->vendor_id  = $user->id;
        $data->status = "SUBMITTED";
        $data->id_active = 0;
        $data->created_by = $user->name;
        $data->save();

        return redirect()->route('vendor.domain_registration')->with('success','Your Domain Registration Success Please Wait For Admin Approval.');
    }

    public function bookingReport(Request $request)
    {
        $data = [
            'bookings'    => $this->bookingClass::getBookingHistory($request->input('status'), false, Auth::id()),
            'statues'     => config('booking.statuses'),
            'breadcrumbs' => [
                [
                    'name'  => __('Booking Report'),
                    'class' => 'active'
                ],
            ],
            'page_title'  => __("Booking Report"),
        ];
        return view('Vendor::frontend.bookingReport.index', $data);
    }

    public function settings(){
        // if(Auth::user() ==null){
        //     return redirect()->route("home");
        // }
        $active_modules = [];
        // get all active modules here
        $custom_modules = ServiceProvider::getModules();
        if(!empty($custom_modules)){
            foreach($custom_modules as $module){
                $moduleClass = "\\Modules\\".ucfirst($module)."\\ModuleProvider";
                if(class_exists($moduleClass))
                {
                    $menuConfigs = call_user_func([$moduleClass,'getUserMenu']);
                    if(!empty($menuConfigs)){
                        foreach($menuConfigs as $menuConfig){
                            if(!empty($menuConfig['permission'])){
                                // push active module in this array
                                $active_modules[] = array_push($active_modules,$menuConfig['permission']);
                            }
                        }
                    }
                }
            }
        }
        // remove some integer item of this array
        $get_only_string_item = array_filter( $active_modules, 
                        function($arrayEntry) { 
                            return !is_numeric($arrayEntry);
                        }
                      );
        // remove duplicate item in this array              
        $active_modules = array_unique($get_only_string_item);
        $vendor_modules = VendorModuleAccess::where('user_id',Auth::user()->id)->first();
        if($vendor_modules == null){
            return redirect()->route("home");
        }
        $data = [
            'active_modules' => $active_modules,
            'vendor_modules' => $vendor_modules,
            'gateways'   => $this->getGateways(),
            'breadcrumbs' => [
                [
                    'name'  => __('Vendor Settings'),
                    'class' => 'active'
                ],
            ],
            'page_title'  => __("Vendor Settings "),
        ];
        return view('Vendor::frontend.settings.index',$data);
    }

    public function accessModuleStore(Request $request){
        $default_permission = array("dashboard_vendor_access");
        $marge_permission = array_merge($default_permission,$request->module_access);
        $vendor_module_access = VendorModuleAccess::where('user_id',Auth::user()->id)->first();
        $vendor_module_access->module_accesses = json_encode($marge_permission);
        $vendor_module_access->save();

        return redirect()->route('vendor.settings.index')->with("success","Module Update Success Full");
    }

    public function getGateways()
    {

        $all = get_payment_gateways();
        $res = [];
        foreach ($all as $k => $item) {
            if (class_exists($item)) {
                $obj = new $item($k);
                if ($obj->isAvailable()) {
                    $res[$k] = $obj;
                }
            }
        }
        return $res;
    }

    public function vendorPaymentMethod(Request $request){
        $default_payment = array("Offline Payment");
        $marge_payment = $request->payment_method != null ?  array_merge($default_payment,$request->payment_method) : $default_payment;
        $vendor_payment_method = VendorModuleAccess::where('user_id',Auth::user()->id)->first();
        $vendor_payment_method->payment_method = json_encode($marge_payment);
        $vendor_payment_method->save();

        return redirect()->route('vendor.settings.index')->with("success","Payment Method Update Success Full");
    }
}